//
//  BiokeyService.swift
//  Demo iOS
//
//  Created by Benoit Vasseur on 13/07/2022.
//

import Foundation
import LocalAuthentication

//https://developer.apple.com/documentation/localauthentication/accessing_keychain_items_with_face_id_or_touch_id
class BiokeyService {

    // MARK: - Singleton
    public static let sharedInstance = BiokeyService()
    
    private let SERVICE_NAME = "Biokey"
    private let ACCOUNT_NAME = "Inwebo"
    
    private init() {
        
    }
    
    func activateBiometrics() async -> Bool {
        var error: NSError?
        let context = LAContext()

        guard context.canEvaluatePolicy(.deviceOwnerAuthenticationWithBiometrics, error: &error) else {
            print(error?.localizedDescription ?? "Can't evaluate policy")
            return false
        }
        
        do {
            return try await context.evaluatePolicy(.deviceOwnerAuthenticationWithBiometrics, localizedReason: "Log in to your account")
        } catch let error {
            print(error.localizedDescription)
            return false
        }
    }
    
    func createBiokey() -> String {
        let uuid = UUID().uuidString
        let context = LAContext()
        context.localizedReason = "Create your biokey in the keychain"

        let access = SecAccessControlCreateWithFlags(nil, // Use the default allocator.
                                                     kSecAttrAccessibleWhenUnlockedThisDeviceOnly,
                                                     .biometryCurrentSet,
                                                     nil) // Ignore any error.
        let keychainQuery = [
            kSecClass: kSecClassGenericPassword,
            kSecAttrService: SERVICE_NAME,
            kSecAttrAccount: ACCOUNT_NAME,
            kSecAttrAccessControl: access as Any,
            kSecUseAuthenticationContext as String: context,
            kSecValueData: uuid.data(using: .utf8)!
        ] as CFDictionary

        SecItemDelete(keychainQuery)
        SecItemAdd(keychainQuery, nil)
        
        return uuid
    }
    
    func fetchBiokey() -> String? {
        
        let context = LAContext()
        context.localizedReason = "Access your biokey on the keychain"
        
        let query = [
            kSecClass: kSecClassGenericPassword,
            kSecAttrService: SERVICE_NAME,
            kSecAttrAccount: ACCOUNT_NAME,
            kSecReturnData: true,
            kSecUseAuthenticationContext as String: context,
            kSecMatchLimit: kSecMatchLimitOne
        ] as CFDictionary
        
        var biokey: CFTypeRef? = nil
        let status = SecItemCopyMatching(query, &biokey)
        
        guard let biokey = biokey, status == errSecSuccess else {
            print("Private key could not be retrieved, status: \(status.description)")
            return nil
        }

        return String(data: biokey as! Data, encoding: .utf8)
    }
    
    func deleteBiokey() -> Bool {
        let query = [
            kSecClass: kSecClassGenericPassword,
            kSecAttrService: SERVICE_NAME,
            kSecAttrAccount: ACCOUNT_NAME,
            kSecReturnData: true,
            kSecMatchLimit: kSecMatchLimitOne
        ] as CFDictionary
        
        let delete = SecItemDelete(query)
        return delete == 0
    }
    
}
